//	Altirra - Atari 800/800XL/5200 emulator
//	Copyright (C) 2024 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License along
//	with this program. If not, see <http://www.gnu.org/licenses/>.

#include <stdafx.h>
#include "printer1020font.h"

namespace nsATPrinter1020Font {
	constexpr uint8 operator""_u(unsigned long long v) {
		return (uint8)v;
	}

	constexpr uint8 operator""_d(unsigned long long v) {
		return (uint8)v | ATPrinterFont1020::kMoveBit;
	}

	constexpr uint8 operator""_end(unsigned long long v) {
		return (uint8)v | ATPrinterFont1020::kEndBit;
	}

	// This data is hand-authored glyph data based on the VIC-1520 manual and
	// firmware disassembly, and comparing to the glyph pictures in the 1020
	// service manual (since the 1020 has a different character set).
	//
	// The VIC-1520 tells us the firmware encoding: an 8x8 grid, of which only
	// 6x8 is actually used. This makes sense, since the step resolution is
	// 0-480 horizontally and this is the most that can fit at 80 columns. It
	// also means that the different text size modes simply apply a uniform
	// integer scale. Although theoretically the format could encode arbitrary
	// sloped lines, the 1020 glyph designs only use 8-way draws. Baseline
	// spacing is 12 units apart, so there are an additional four rows between
	// character cells of adjacent lines.
	//
	// The encoding used here is similar to the VIC-1520 firmware encoding, but
	// with different bitfields so the XY coordinates are visible in hex. The
	// extra two bits are used to signal pen up/down and end; only pen down
	// is actually encoded as it is used to force a move-to, and up is not
	// redundant. Blank glyphs use the special case of a single 0x00_end.
	//
	// The coordinate system is bottom-up with (0,0) being at the insertion
	// point on the baseline. This means that the descenders are above the
	// baseline. The 1020 diagnostic test actually requires this as the
	// positioning that it does between drawing S characters of different scale
	// requires the base of the S to shift with the scale.
	//
	// The glyphs here were hand-authored directly to coordinates by eyeballing.
	// The shapes should be accurate, but the stroke pattern/order almost
	// certainly isn't. This currently doesn't matter as the 1020 emulator
	// doesn't emulate draw timing. The glyphs are designed without overlapping
	// strokes.
	//
	static constexpr uint8 kPrinterFontData1020[] {
		/* $00 a' */	0x16_d, 0x27_u,
						0x05_d, 0x25, 0x34, 0x32_u, 0x33_d, 0x24, 0x14, 0x03, 0x02, 0x11, 0x21, 0x32, 0x41_end,
		/* $01 u' */	0x26_d, 0x17_u, 
						0x05_d, 0x02, 0x11, 0x21, 0x32, 0x35_u, 0x32_d, 0x41_end,
		/* $02 N~ */	0x06_d, 0x17, 0x26, 0x37_u,
						0x01_d, 0x05, 0x41, 0x45_end,
		/* $03 E' */	0x16_d, 0x27_u, 0x45_d, 0x05, 0x03, 0x33_u, 0x03_d, 0x01, 0x41_end,
		/* $04 C, */	0x46_d, 0x37, 0x17, 0x06, 0x04, 0x13, 0x33, 0x44_u, 0x23_d, 0x12, 0x32, 0x21_end,
		/* $05 o^ */	0x06_d, 0x17, 0x26_u,
						0x25_d, 0x15, 0x04, 0x02, 0x11, 0x21, 0x32, 0x34, 0x25_end,
		/* $06 o` */	0x17_d, 0x26_u,
						0x25_d, 0x15, 0x04, 0x02, 0x11, 0x21, 0x32, 0x34, 0x25_end,
		/* $07 i` */	0x17_d, 0x26_u,
						0x25_d, 0x15_u, 0x14_d, 0x24, 0x21_u, 0x11_d, 0x31_end,
		/* $08 Lb */	0x11_d, 0x15, 0x06, 0x17, 0x27, 0x36_u, 0x24_d, 0x04_u, 0x01_d, 0x31, 0x42_end,
		/* $09 i" */	0x07_d, 0x06, 0x16, 0x17, 0x07_u, 0x37_d, 0x47, 0x46, 0x36, 0x37_u,
						0x25_d, 0x15_u, 0x14_d, 0x24, 0x21_u, 0x11_d, 0x31_end,
		/* $0A u" */	0x07_d, 0x06, 0x16, 0x17, 0x07_u, 0x37_d, 0x47, 0x46, 0x36, 0x37_u,
						0x05_d, 0x02, 0x11, 0x21, 0x32, 0x35_u, 0x32_d, 0x41_end,
		/* $0B a" */	0x07_d, 0x06, 0x16, 0x17, 0x07_u, 0x37_d, 0x47, 0x46, 0x36, 0x37_u,
						0x05_d, 0x25, 0x34, 0x32_u, 0x33_d, 0x24, 0x14, 0x03, 0x02, 0x11, 0x21, 0x32, 0x41_end,
		/* $0C O" */	0x07_d, 0x06, 0x16, 0x17, 0x07_u, 0x37_d, 0x47, 0x46, 0x36, 0x37_u,
						0x44_d, 0x35, 0x15, 0x04, 0x02, 0x11, 0x31, 0x42, 0x44_end,
		/* $0D u' */	0x16_d, 0x27_u,
						0x05_d, 0x02, 0x11, 0x21, 0x32, 0x35_u, 0x32_d, 0x41_end,
		/* $0E o' */	0x16_d, 0x27_u,
						0x25_d, 0x15, 0x04, 0x02, 0x11, 0x21, 0x32, 0x34, 0x25_end,
		/* $0F o" */	0x07_d, 0x06, 0x16, 0x17, 0x07_u, 0x27_d, 0x37, 0x36, 0x26, 0x27_u, 
						0x25_d, 0x15, 0x04, 0x02, 0x11, 0x21, 0x32, 0x34, 0x25_end,
		/* $10 U" */	0x07_d, 0x06, 0x16, 0x17, 0x07_u, 0x37_d, 0x47, 0x46, 0x36, 0x37_u,
						0x05_d, 0x02, 0x11, 0x31, 0x42, 0x45_end,
		/* $11 a^ */	0x16_d, 0x27, 0x36_u,
						0x05_d, 0x25, 0x34, 0x32_u, 0x33_d, 0x24, 0x14, 0x03, 0x02, 0x11, 0x21, 0x32, 0x41_end,
		/* $12 u^ */	0x16_d, 0x27, 0x36_u,
						0x05_d, 0x02, 0x11, 0x21, 0x32, 0x35_u, 0x32_d, 0x41_end,
		/* $13 i^ */	0x16_d, 0x27, 0x36_u,
						0x25_d, 0x15_u, 0x14_d, 0x24, 0x21_u, 0x11_d, 0x31_end,
		/* $14 e' */	0x16_d, 0x27_u,
						0x03_d, 0x33, 0x44, 0x35, 0x15, 0x04, 0x02, 0x11, 0x31, 0x42_end,
		/* $15 e` */	0x17_d, 0x26_u,
						0x03_d, 0x33, 0x44, 0x35, 0x15, 0x04, 0x02, 0x11, 0x31, 0x42_end,
		/* $16 n~ */	0x06_d, 0x17, 0x26, 0x37_u,
						0x05_d, 0x01_u, 0x04_d, 0x15, 0x35, 0x44, 0x41_end,
		/* $17 e^ */	0x16_d, 0x27, 0x36_u,
						0x03_d, 0x33, 0x44, 0x35, 0x15, 0x04, 0x02, 0x11, 0x31, 0x42_end,
		/* $18 a* */	0x16_d, 0x17, 0x27, 0x26, 0x16_u,
						0x05_d, 0x25, 0x34, 0x32_u, 0x33_d, 0x24, 0x14, 0x03, 0x02, 0x11, 0x21, 0x32, 0x41_end,
		/* $19 a` */	0x17_d, 0x26_u,
						0x05_d, 0x25, 0x34, 0x32_u, 0x33_d, 0x24, 0x14, 0x03, 0x02, 0x11, 0x21, 0x32, 0x41_end,
		/* $1A A* */	0x16_d, 0x17, 0x27, 0x26, 0x16_u,
						0x02_d, 0x42_u, 0x01_d, 0x03, 0x25, 0x43, 0x41_end,
		/* $1B esc*/	0x00_end,
		/* $1C up */	0x05_d, 0x27, 0x45_u, 0x27_d, 0x21_end,
		/* $1D dn */	0x03_d, 0x21, 0x43_u, 0x27_d, 0x21_end,
		/* $1E lf */	0x26_d, 0x04, 0x22_u, 0x04_d, 0x44_end,
		/* $1F rt */	0x04_d, 0x44_u, 0x26_d, 0x44, 0x22_end,
		/* $20 sp */	0x00_end,
		/* $21 !  */	0x11_d, 0x22, 0x31, 0x11_u, 0x23_d, 0x14, 0x17, 0x37, 0x34, 0x23_end,
		/* $22 "  */	0x17_d, 0x16_u, 0x37_d, 0x36_end,
		/* $23 #  */	0x11_d, 0x17_u, 0x37_d, 0x31_u, 0x43_d, 0x03_u, 0x05_d, 0x45_end,
		/* $24 $  */	0x03_d, 0x12, 0x32, 0x43, 0x34, 0x14, 0x05, 0x16, 0x36, 0x45_u, 0x27_d, 0x21_end,
		/* $25 %  */	0x02_d, 0x46_u, 0x26_d, 0x16, 0x15, 0x25, 0x26_u, 0x33_d, 0x32, 0x22, 0x23, 0x33_end,
		/* $26 &  */	0x43_d, 0x21, 0x11, 0x02, 0x03, 0x25, 0x26, 0x17, 0x06, 0x05, 0x41_end,
		/* $27 '  */	0x14_d, 0x25, 0x27, 0x17, 0x16, 0x26_end,
		/* $28 (  */	0x37_d, 0x27, 0x16, 0x12, 0x21, 0x31_end,
		/* $29 )  */	0x17_d, 0x27, 0x36, 0x32, 0x21, 0x11_end,
		/* $2A *  */	0x02_d, 0x46_u, 0x42_d, 0x06_u, 0x27_d, 0x21_end,
		/* $2B +  */	0x26_d, 0x22_u, 0x04_d, 0x44_end,
		/* $2C ,  */	0x22_d, 0x12, 0x13, 0x23, 0x21, 0x10_end,
		/* $2D -  */	0x04_d, 0x44_end,
		/* $2E .  */	0x11_d, 0x12, 0x22, 0x21, 0x11_end,
		/* $2F /  */	0x12_d, 0x46_end,
		/* $30 0  */	0x02_d, 0x46, 0x37, 0x17, 0x06, 0x02, 0x11, 0x31, 0x42, 0x46_end,
		/* $31 1  */	0x16_d, 0x27, 0x21_u, 0x11_d, 0x31_end,
		/* $32 2  */	0x06_d, 0x17, 0x37, 0x46, 0x45, 0x01, 0x41_end,
		/* $33 3  */	0x06_d, 0x17, 0x37, 0x46, 0x45, 0x34, 0x24_u, 0x34_d, 0x43, 0x42, 0x31, 0x11, 0x02_end,
		/* $34 4  */	0x31_d, 0x37, 0x04, 0x03, 0x43_end,
		/* $35 5  */	0x02_d, 0x11, 0x31, 0x42, 0x44, 0x35, 0x05, 0x07, 0x47_end,
		/* $36 6  */	0x04_d, 0x15, 0x35, 0x44, 0x42, 0x31, 0x11, 0x02, 0x06, 0x17, 0x37, 0x46_end,
		/* $37 7  */	0x07_d, 0x47, 0x45, 0x01_end,
		/* $38 8  */	0x06_d, 0x17, 0x37, 0x46, 0x45, 0x34, 0x14, 0x05, 0x06_u, 0x14_d, 0x03, 0x02, 0x11, 0x31, 0x42, 0x43, 0x34, 0x14_end,
		/* $39 9  */	0x02_d, 0x11, 0x31, 0x42, 0x46, 0x37, 0x17, 0x06, 0x05, 0x14, 0x34, 0x45_end,
		/* $3A :  */	0x26_d, 0x36, 0x35, 0x25, 0x26_u, 0x23_d, 0x33, 0x32, 0x22, 0x23_end,
		/* $3B ;  */	0x26_d, 0x36, 0x35, 0x25, 0x26_u, 0x32_d, 0x22, 0x23, 0x33, 0x31, 0x20_end,
		/* $3C <  */	0x47_d, 0x14, 0x41_end,
		/* $3D =  */	0x03_d, 0x43_u, 0x45_d, 0x05_end,
		/* $3E >  */	0x01_d, 0x34, 0x07_end,
		/* $3F ?  */	0x05_d, 0x06, 0x17, 0x37, 0x46, 0x45, 0x34, 0x24, 0x23_u, 0x22_d, 0x21_end,
		/* $40 @  */	0x33_d, 0x35, 0x25, 0x14, 0x13, 0x22, 0x33, 0x43, 0x46, 0x37, 0x17, 0x06, 0x02, 0x11, 0x41_end,
		/* $41 A  */	0x01_d, 0x05, 0x27, 0x45, 0x41_u, 0x04_d, 0x44_end,
		/* $42 B  */	0x01_d, 0x07, 0x37, 0x46, 0x45, 0x34, 0x04_u, 0x34_d, 0x43, 0x42, 0x31, 0x01_end,
		/* $43 C  */	0x46_d, 0x37, 0x17, 0x06, 0x02, 0x11, 0x31, 0x42_end,
		/* $44 D  */	0x01_d, 0x07, 0x37, 0x46, 0x42, 0x31, 0x01_end,
		/* $45 E  */	0x41_d, 0x01, 0x07, 0x47_u, 0x04_d, 0x34_end,
		/* $46 F  */	0x01_d, 0x07, 0x47_u, 0x04_d, 0x34_end,
		/* $47 G  */	0x24_d, 0x44, 0x41, 0x11, 0x02, 0x06, 0x17, 0x37, 0x46_end,
		/* $48 H  */	0x01_d, 0x07_u, 0x04_d, 0x44_u, 0x47_d, 0x41_end,
		/* $49 I  */	0x07_d, 0x47_u, 0x27_d, 0x21_u, 0x01_d, 0x41_end,
		/* $4A J  */	0x02_d, 0x11, 0x21, 0x32, 0x37_end,
		/* $4B K  */	0x01_d, 0x07_u, 0x03_d, 0x47_u, 0x14_d, 0x41_end,
		/* $4C L  */	0x07_d, 0x01, 0x41_end,
		/* $4D M  */	0x01_d, 0x07, 0x25, 0x47, 0x41_end,
		/* $4E N  */	0x01_d, 0x07_u, 0x06_d, 0x42_u, 0x41_d, 0x47_end,
		/* $4F O  */	0x02_d, 0x06, 0x17, 0x37, 0x46, 0x42, 0x31, 0x11, 0x02_end,
		/* $50 P  */	0x01_d, 0x07, 0x37, 0x46, 0x45, 0x34, 0x04_end,
		/* $51 Q  */	0x02_d, 0x06, 0x17, 0x37, 0x46, 0x42, 0x31, 0x11, 0x02_u, 0x23_d, 0x41_end,
		/* $52 R  */	0x01_d, 0x07, 0x37, 0x46, 0x45, 0x34, 0x04_u, 0x14_d, 0x41_end,
		/* $53 S  */	0x02_d, 0x11, 0x31, 0x42, 0x43, 0x34, 0x14, 0x05, 0x06, 0x17, 0x37, 0x46_end,
		/* $54 T  */	0x07_d, 0x47_u, 0x27_d, 0x21_end,
		/* $55 U  */	0x07_d, 0x02, 0x11, 0x31, 0x42, 0x47_end,
		/* $56 V  */	0x07_d, 0x03, 0x21, 0x43, 0x47_end,
		/* $57 W  */	0x07_d, 0x01, 0x23, 0x24_u, 0x23_d, 0x41, 0x47_end,
		/* $58 X  */	0x01_d, 0x02, 0x46, 0x47_u, 0x07_d, 0x06, 0x42, 0x41_end,
		/* $59 Y  */	0x07_d, 0x06, 0x24, 0x46, 0x47_u, 0x24_d, 0x21_end,
		/* $5A Z  */	0x33_d, 0x15_u, 0x07_d, 0x37, 0x46, 0x02, 0x11, 0x41_end,
		/* $5B [  */	0x37_d, 0x17, 0x11, 0x31_end,
		/* $5C \  */	0x06_d, 0x42_end,
		/* $5D ]  */	0x17_d, 0x37, 0x31, 0x11_end,
		/* $5E ^  */	0x04_d, 0x26, 0x44_end,
		/* $5F _  */	0x00_d, 0x60_end,
		/* $60 i* */	0x17_d, 0x27, 0x26, 0x16, 0x17_u,
						0x15_d, 0x35_u, 0x25_d, 0x21_u, 0x11_d, 0x31_end,
		/* $61 a  */	0x05_d, 0x25, 0x34, 0x32_u, 0x33_d, 0x24, 0x14, 0x03, 0x02, 0x11, 0x21, 0x32, 0x41_end,
		/* $62 b  */	0x07_d, 0x01, 0x21, 0x32, 0x34, 0x25, 0x05_end,
		/* $63 c  */	0x34_d, 0x25, 0x15, 0x04, 0x02, 0x11, 0x21, 0x32_end,
		/* $64 d  */	0x35_d, 0x15, 0x04, 0x02, 0x11, 0x31, 0x37_end,
		/* $65 e  */	0x03_d, 0x33, 0x34, 0x25, 0x15, 0x04, 0x02, 0x11, 0x31_end,
		/* $66 f  */	0x14_d, 0x34_u, 0x37_d, 0x27, 0x21_end,
		/* $67 g  */	0x00_d, 0x20, 0x31, 0x34, 0x25, 0x15, 0x04, 0x03, 0x12, 0x22, 0x33_end,
		/* $68 h  */	0x01_d, 0x07_u, 0x05_d, 0x25, 0x34, 0x31_end,
		/* $69 i  */	0x21_d, 0x24_u, 0x25_d, 0x26_end,
		/* $6A j  */	0x01_d, 0x10, 0x20, 0x31, 0x34_u, 0x35_d, 0x36_end,
		/* $6B k  */	0x01_d, 0x07_u, 0x02_d, 0x35_u, 0x13_d, 0x31_end,
		/* $6C l  */	0x27_d, 0x22, 0x31_end,
		/* $6D m  */	0x01_d, 0x05_u, 0x04_d, 0x15, 0x24, 0x21_u, 0x24_d, 0x35, 0x44, 0x41_end,
		/* $6E n  */	0x01_d, 0x05_u, 0x04_d, 0x15, 0x25, 0x34, 0x31_end,
		/* $6F o  */	0x02_d, 0x04, 0x15, 0x25, 0x34, 0x32, 0x21, 0x11, 0x02_end,
		/* $70 p  */	0x00_d, 0x05, 0x25, 0x34, 0x33, 0x22, 0x02_end,
		/* $71 q  */	0x32_d, 0x12, 0x03, 0x04, 0x15, 0x35, 0x30, 0x40_end,
		/* $72 r  */	0x05_d, 0x15, 0x11_u, 0x14_d, 0x25, 0x35_end,
		/* $73 s  */	0x02_d, 0x11, 0x21, 0x32, 0x23, 0x13, 0x04, 0x15, 0x25, 0x34_end,
		/* $74 t  */	0x05_d, 0x25_u, 0x17_d, 0x11, 0x21, 0x32_end,
		/* $75 u  */	0x05_d, 0x02, 0x11, 0x21, 0x32, 0x35_u, 0x32_d, 0x41_end,
		/* $76 v  */	0x05_d, 0x03, 0x21, 0x43, 0x45_end,
		/* $77 w  */	0x05_d, 0x02, 0x11, 0x22, 0x23_u, 0x22_d, 0x31, 0x42, 0x45_end,
		/* $78 x  */	0x01_d, 0x45_u, 0x05_d, 0x41_end,
		/* $79 y  */	0x05_d, 0x04, 0x22_u, 0x45_d, 0x44, 0x00_end,
		/* $7A z  */	0x05_d, 0x45, 0x01, 0x41_end,
		/* $7B A" */	0x07_d, 0x06, 0x16, 0x17, 0x07_u, 0x37_d, 0x36, 0x46, 0x47, 0x37_u,
						0x42_d, 0x02_u, 0x01, 0x03, 0x25, 0x43, 0x41_end,
		/* $7C |  */	0x21_d, 0x27_end,
		/* $7D hm */	0x04_d, 0x07, 0x37_u, 0x07_d, 0x34, 0x31_end,
		/* $7E << */	0x04_d, 0x37, 0x31, 0x04_end,
		/* $7F >> */	0x11_d, 0x17, 0x44, 0x11_end
	};
}

extern const constinit ATPrinterFont1020 g_ATPrinterFont1020 = []() {
	ATPrinterFont1020 font {};

	const uint8 *p = nsATPrinter1020Font::kPrinterFontData1020;
	font.mpFontData = p;

	size_t offset = 0;

	// index the starting offsets of all characters
	for(int i=0; i<128; ++i) {
		font.mCharOffsets[i] = offset;

		while(!(p[offset++] & 0x80))
			;
	}

	return font;
} ();
